#!/usr/bin/env bash
set -euo pipefail

# -----------------------------
# Логування
# -----------------------------
log(){  printf "\033[1;32m[OK]\033[0m %s\n"   "$*"; }
info(){ printf "\033[1;34m[INFO]\033[0m %s\n" "$*"; }
warn(){ printf "\033[1;33m[WARN]\033[0m %s\n" "$*"; }

need_sudo(){
  if [ "${EUID:-$(id -u)}" -ne 0 ]; then
    warn "Деякі кроки потребують sudo. Вас попросять ввести пароль."
  fi
}

# -----------------------------
# Визначення ОС / менеджера пакетів
# -----------------------------
OS="$(uname -s)"
PM=""
APT_OPTS=()
UPDATE_CMD=()

case "$OS" in
  Linux)
    if command -v apt-get >/dev/null 2>&1; then
      PM="apt"
      # VARIANT 1: лише офіційний /etc/apt/sources.list
      APT_OPTS=( -o Dir::Etc::sourceparts=/dev/null -o Dir::Etc::sourcelist=/etc/apt/sources.list )
      UPDATE_CMD=( sudo apt-get "${APT_OPTS[@]}" update -y )
    elif command -v dnf >/dev/null 2>&1; then
      PM="dnf";    UPDATE_CMD=( sudo dnf -y makecache )
    elif command -v yum >/dev/null 2>&1; then
      PM="yum";    UPDATE_CMD=( sudo yum -y makecache )
    elif command -v zypper >/dev/null 2>&1; then
      PM="zypper"; UPDATE_CMD=( sudo zypper refresh )
    elif command -v pacman >/dev/null 2>&1; then
      PM="pacman"; UPDATE_CMD=( sudo pacman -Sy --noconfirm )
    else
      echo "Непідтримуваний дистрибутив Linux"; exit 1
    fi
    ;;
  *) echo "Непідтримувана ОС: $OS"; exit 1;;
esac

need_sudo

# -----------------------------
# Універсальна інсталяція пакету
# -----------------------------
install_pkg() {
  local pkg="$1"
  case "$PM" in
    apt)    sudo apt-get "${APT_OPTS[@]}" install -y "$pkg" ;;
    dnf)    sudo dnf install -y "$pkg" ;;
    yum)    sudo yum install -y "$pkg" ;;
    zypper) sudo zypper install -y "$pkg" ;;
    pacman) sudo pacman -S --noconfirm "$pkg" ;;
    *) echo "Unknown package manager"; return 1 ;;
  esac
}

# -----------------------------
# Оновлення індексів пакетів (лише офіційні джерела)
# -----------------------------
"${UPDATE_CMD[@]}"

# -----------------------------
# Базові тулінги для збірок
# -----------------------------
install_pkg pkg-config || true
install_pkg build-essential || install_pkg @development-tools || true
install_pkg ca-certificates || true
install_pkg gnupg || true
install_pkg software-properties-common || true
install_pkg curl || true

# -----------------------------
# APT-хелпери (Variant 1)
# -----------------------------
apt_has_candidate() {
  local pkg="$1"
  apt-cache policy "$pkg" 2>/dev/null | grep -q "Candidate:" && \
  ! apt-cache policy "$pkg" | grep -q "Candidate: (none)"
}

ensure_deadsnakes_in_sources_list() {
  local keyring="/usr/share/keyrings/deadsnakes.gpg"
  local line='deb [signed-by=/usr/share/keyrings/deadsnakes.gpg] http://ppa.launchpad.net/deadsnakes/ppa/ubuntu jammy main'
  if ! grep -q "^deb .*deadsnakes/ppa" /etc/apt/sources.list 2>/dev/null; then
    info "Додаю Deadsnakes PPA в /etc/apt/sources.list (jammy)…"
    sudo mkdir -p "$(dirname "$keyring")"
    curl -fsSL https://ppa.launchpadcontent.net/deadsnakes/ppa/ubuntu/KEY.gpg \
      | sudo gpg --dearmor -o "$keyring"
    sudo tee -a /etc/apt/sources.list >/dev/null <<<"$line"
    sudo apt-get "${APT_OPTS[@]}" update -y
  fi
}

# -----------------------------
# Встановлення Python 3.12 (+dev, +venv, +virtualenv як fallback)
# -----------------------------
if ! command -v python3.12 >/dev/null 2>&1; then
  case "$PM" in
    apt)
      if ! apt_has_candidate "python3.12"; then
        ensure_deadsnakes_in_sources_list
      fi
      install_pkg python3.12 || true
      install_pkg python3.12-dev || install_pkg python3-dev || true
      install_pkg python3.12-venv || install_pkg python3-venv || true
      install_pkg python3-virtualenv || true
      ;;
    dnf|yum)
      install_pkg python3.12 || install_pkg python3 || true
      install_pkg python3.12-devel || install_pkg python3-devel || true
      install_pkg python3.12-venv || install_pkg python3-virtualenv || install_pkg python-virtualenv || true
      ;;
    zypper)
      install_pkg python312 || install_pkg python3 || true
      install_pkg python312-devel || install_pkg python3-devel || true
      install_pkg python312-venv || install_pkg python3-virtualenv || true
      ;;
    pacman)
      install_pkg python || true
      install_pkg python-virtualenv || true
      ;;
  esac
else
  # Навіть якщо python3.12 вже є — гарантуємо наявність venv/virtualenv
  case "$PM" in
    apt)    install_pkg python3.12-venv || install_pkg python3-venv || true; install_pkg python3-virtualenv || true ;;
    dnf|yum) install_pkg python3.12-venv || install_pkg python3-virtualenv || true ;;
    zypper) install_pkg python312-venv || install_pkg python3-virtualenv || true ;;
    pacman) install_pkg python-virtualenv || true ;;
  esac
fi

PYBIN="$(command -v python3.12 || true)"
if [ -z "${PYBIN}" ]; then
  echo "Python 3.12 не знайдено після інсталяції."; exit 1
fi
log "Використовую: $("$PYBIN" -V)"

# -----------------------------
# pip для Python 3.12 (глобально)
# -----------------------------
"$PYBIN" -m ensurepip --upgrade || true
"$PYBIN" -m pip install --upgrade pip setuptools wheel || true

# -----------------------------
# dev-залежності для MySQL/PostgreSQL
# -----------------------------
info "Ставлю dev-бібліотеки для MySQL/PostgreSQL…"
case "$PM" in
  apt)
    install_pkg default-libmysqlclient-dev || install_pkg libmysqlclient-dev || true
    install_pkg libpq-dev || true
    ;;
  dnf|yum)
    install_pkg mariadb-connector-c-devel || install_pkg mysql-devel || true
    install_pkg postgresql-devel || true
    ;;
  zypper)
    install_pkg libmysqlclient-devel || install_pkg mariadb-connector-c-devel || true
    install_pkg postgresql-devel || true
    ;;
  pacman)
    install_pkg mariadb-libs || true
    install_pkg postgresql-libs || true
    ;;
esac

log "check_python.sh завершено."

